// If you're using (parts of) this work, please cite the corresponding
// publication: ivrg.epfl.ch/Lindner_IEEE_MM_2015
//
// For any questions contact the author:
// ajl.epfl@gmail.com  http://ivrg.epfl.ch/people/lindner

#include "mex.h"
#include <math.h>

inline double sRGB2linRGB(double in) {
    if (in <= 0.03928) {
        return in/12.92;
	}
	else {
        return pow( (in+0.055)/1.055, 2.4 ); 
	}
}

inline double f(double in) {
    if (in <= 0.00885645168) {
        return 7.78703704*in + 0.137931034;
	}
	else {
        return pow( in, 1/3.0 ); 
	}
}

/*
 The conversion from sRGB to XYZ is based on the formulas in:
 Michael Stokes, Matthew Anderson, Srinivasan Chandrasekar, Ricardo Motta (1996). "A Standard Default Color Space for the Internet - sRGB"
 The conversion from XYZ to CIELAB is based on the formulas in:
 Robert Hunt, "Measuring Color", 3rd edition, 1998
 
 see als: http://www.brucelindbloom.com/
*/
void sRGB2Lab(double *sRGB, double *Lab, int H, int W) {
    int HW = H*W;
    
    /* white point D65*/
    const double Xn = 0.9504;
    const double Yn = 1.0;
    const double Zn = 1.0889;
    
    for (int i = 0; i<HW; i++) {
        //mexPrintf("%d\n", i);
        double R = sRGB[i];
        double G = sRGB[i+HW];
        double B = sRGB[i+2*HW];
        
        if (R>1 || G>1 || B>1) {
            mexErrMsgTxt("sRGB input image has to be in the interval [0 1].\n");
        }
        //mexPrintf("sRGB: %f %f %f\n", R, G, B);
        
        /* linearize rgb values */
        R = sRGB2linRGB(R);
        G = sRGB2linRGB(G);
        B = sRGB2linRGB(B);
        
        //mexPrintf("linRGB: %f %f %f\n", R, G, B);
        
        /* convert to XYZ */
        double X = 0.4124*R + 0.3576*G + 0.1805*B;
        double Y = 0.2126*R + 0.7152*G + 0.0722*B;
        double Z = 0.0193*R + 0.1192*G + 0.9505*B;
        
        //mexPrintf("XYZ: %f %f %f\n", X, Y, Z);
        
        /* convert to CIALB */
        //mexPrintf("ratios: %f %f %f\n", X/Xn, Y/Yn, Z/Zn);
        //mexPrintf("f(ratios): %f %f %f\n", f(X/Xn), f(Y/Yn), f(Z/Zn));
        
        Lab[i] = 116*f(Y/Yn) - 16;
        Lab[i+HW] = 500*( f(X/Xn) - f(Y/Yn) );
        Lab[i+2*HW] = 200*( f(Y/Yn) - f(Z/Zn) );
        
        //mexPrintf("Lab: %f %f %f\n", Lab[i], Lab[i+HW], Lab[i+2*HW]);
        //mexPrintf("\n");
    }
}


void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
    double *sRGB;
    double *Lab;
    int H, W, D, HW;
    mwSize number_of_dimensions; 
    const mwSize *dims;
    
    /* check input */
    if(nrhs!=1){
        mexErrMsgTxt("Exactly one input argument required.\n");
    }
    
    /* refuse anything but doubles */
    if (!mxIsDouble(prhs[0])) {
        mexErrMsgTxt("Input data has to be double.\n");
    }
    
    /* get dimensions of image */
    dims = mxGetDimensions(prhs[0]);
    number_of_dimensions = mxGetNumberOfDimensions(prhs[0]);
    
    if (number_of_dimensions == 2) {
        H = dims[0];
        W = 1;
        D = dims[1];
    }
    else if (number_of_dimensions == 3) {
        H = dims[0];
        W = dims[1];
        D = dims[2];
    }
    else {
        mexErrMsgTxt("sRGB input image has to be a [H, W, 3] matrix or a [H*W 3] matrix.\n");
    }
    
    if (D != 3) {
        mexErrMsgTxt("sRGB input image has to be a [H, W, 3] matrix or a [H*W 3] matrix.\n");
    }
    
    HW = H*W;
    //mexPrintf("H=%d W=%d D=%d HW=%d\n", H, W, D, HW);
    
    
    
    /* input ok, start processing*/
    sRGB = mxGetPr(prhs[0]);
    if (number_of_dimensions == 2) {
        plhs[0] = mxCreateNumericArray(2, dims, mxDOUBLE_CLASS, mxREAL);
    }
	else {
        plhs[0] = mxCreateNumericArray(3, dims, mxDOUBLE_CLASS, mxREAL);
    }
            
    Lab = mxGetPr(plhs[0]);
    
    
    sRGB2Lab(sRGB, Lab, H, W);
    
    return;
}


