% This script takes a significance distribution of one color and displays a
% 3D plot with three slices through the distribution's maximum.
% 
% If you're using (parts of) this work, please cite the corresponding
% publication: ivrg.epfl.ch/Lindner_IEEE_MM_2015

% For any questions contact the author:
% ajl.epfl@gmail.com  http://ivrg.epfl.ch/people/lindner

load ground_truth.mat
FontSize = 18;

[SRGB, LAB] = bincenters('lab_hist15_80');
N = 15;
LabL = reshape(LAB(:,1), [N N N]);
LabA = reshape(LAB(:,2), [N N N]);
LabB = reshape(LAB(:,3), [N N N]);

cName = 'pink';
language = 'en';
view_3d = [150 30];


%% load z distribution
fname = ['./DB/lang_' language '/' cName '.mat'];
load(fname);

%% find maximum bin and associated color values
[A0, idx] = max(zvalues);
bidx = floor((idx-1) / N^2);
aidx = idx - bidx*N^2;
aidx = floor((aidx-1) / N);
Lidx = idx - bidx*N^2 - aidx*N;

aidx = aidx + 1;
bidx = bidx + 1;

Lab_max = LAB(idx, :);
RGB_max = SRGB(idx, :);

fprintf('\nmaximum bin color values for %s (%s):\n', cName, language);
fprintf('RGB = (%d, %d, %d)\n', round(255*RGB_max(1)), round(255*RGB_max(2)), round(255*RGB_max(3)))
fprintf('LAB = (%d, %d, %d)\n', round(Lab_max(1)), round(Lab_max(2)), round(Lab_max(3)));


%% do a fancy 3D-plot of the distribution
L0 = Lab_max(1);
a0 = Lab_max(2);
b0 = Lab_max(3);

colorsA = LabA(1, :, :);
colorsB = LabB(1, :, :);
colorsL = L0*ones(size(colorsA));
colorsLAB = [colorsL(:) colorsA(:) colorsB(:)];
colorsRGB = mexLab2sRGB(colorsLAB);
colorsRGB = reshape(colorsRGB, [N N 3]);

% display a color plane at the bottom (L=5) of the plot for better
% orientation in CIELAB
close all
fh = figure(1);
hold off
surf(squeeze(colorsA), squeeze(colorsB), 5*ones(N), colorsRGB, 'facecolor', 'texture', 'LineStyle', 'none');
hold on


L = LabL(:, 1, 1);
a = LabA(1, :, 1);
b = LabB(1, 1, :);
minL = min(L); maxL = max(L);
mina = min(a); maxa = max(a);
minb = min(b); maxb = max(b);

% clean up data to make plot look nice
minz = min(zvalues);
zz = double(reshape(zvalues, [N N N]));
zz(zz==Inf) = minz;
zz(imag(zz)>1e-6) = minz;
zz(zz==0) = minz;
zz = permute(zz, [3 2 1]);

% plot three slices
[meshA, meshB, meshL] = meshgrid(a, b, L);
slice(meshA, meshB, meshL, zz, a0, b0, L0, 'spline');

% plot white lines along plane intersections
plot3([a0 a0], [b0 b0], [minL maxL], 'w', 'LineWidth', 2);
plot3([a0 a0], [minb maxb], [L0 L0], 'w', 'LineWidth', 2);
plot3([mina maxa], [b0 b0], [L0 L0], 'w', 'LineWidth', 2);

% format plot
colorbar('location', 'EastOutside')
colormap gray

xlabel('a', 'FontSize', FontSize);
ylabel('b', 'FontSize', FontSize);
zlabel('L', 'FontSize', FontSize);
xlim([-80 80]);
ylim([-80 80]);
zlim([0 100]);
set(gca, 'XTick', [-80 -40 0 40 80], 'FontSize', FontSize);
set(gca, 'YTick', [-80 -40 0 40 80], 'FontSize', FontSize);
set(gca, 'ZTick', [0 20 40 60 80 100], 'FontSize', FontSize);

if mean(RGB_max) > 0.5
    title(sprintf('%s, %s: (%d, %d, %d)', cName, language, round(255*RGB_max(1)), round(255*RGB_max(2)), round(255*RGB_max(3))), 'BackgroundColor', RGB_max, 'Color', 'black');
else
    title(sprintf('%s, %s: (%d, %d, %d)', cName, language, round(255*RGB_max(1)), round(255*RGB_max(2)), round(255*RGB_max(3))), 'BackgroundColor', RGB_max, 'Color', 'white');
end

view(view_3d);
