%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% Code Name  : Realistic character image generation
%%%%
%%%% Coded By   : Gokhan Yildirim, EPFL/IC/IVRG
%%%%
%%%% Description: This code generates realistic cut-out images for digits
%%%% and letters for text detection and recognition purposes. The user
%%%% should supply a database path name, where the created images are
%%%% recorded, and a directory that includes background images. All
%%%% parameters are under "Parameters" section and self-explanatory. If you
%%%% have any questions, please contact to gokhan.yildirim@epfl.ch. This
%%%% code can be used, distributed, and modified by citing the original
%%%% paper (TEXT RECOGNITION IN NATURAL IMAGES USING MULTICLASS HOUGH
%%%% FORESTS).


clc
close all
clear all

global frequency comb

load frequency

fontlist = listfonts;

%%%% Parameters

database_path_name              = '/Users/gokhan/Desktop/trainExamples/';

number_of_examples_per_letter   = 10;

background_images_path_name     = '/Users/gokhan/Downloads/datasets/TrainVal/VOCdevkit/VOC2011/JPEGImages/';

maximum_x_rotation              = 1;
maximum_y_rotation              = 1;
maximum_z_rotation              = 1;

training_image_height           = 24;
training_image_width            = 24;

%%%% CLEARING and re-creating directories for digits and letters

rmdir(database_path_name,'s');

all_characters = [char(48:57) char(65:90) char(97:122)];

for i = 1:length(all_characters)
    if i <= 36
        mkdir([database_path_name all_characters(i)]);
    else
        mkdir([database_path_name all_characters(i) all_characters(i)]);
    end
end

%%%% Load background images

background_image_list = dir([background_images_path_name '*.jpg']);

background_images = cell(length(background_image_list),1);

number_of_background_images = 100;

randomly_selected_images = rand(1,length(background_image_list));

[sorted randomly_selected_images] = sort(randomly_selected_images);

for i = 1:number_of_background_images
    
    image_number = randomly_selected_images(i);
    
    bg = imread([background_images_path_name background_image_list(image_number).name]);
    bg = rgb2gray(bg);
    background_images{i} = bg;
    
end

screen_size = get(0,'screensize');

h = figure('color', 'w', 'resize','off', 'position',[screen_size(3)/2 - 50 screen_size(4)/2 - 50 100 100], 'PaperPositionMode', 'auto');
axis off;

%%%% Over all symbols, running a loop to create necessary traning examples

for letter = [48:57 65:90 97:122]
    
    count = 1;
    
    for t = 1:number_of_examples_per_letter
        
        disp(['Creating examples for symbol: "' char(letter) '" ' '(' num2str(t) '/' num2str(number_of_examples_per_letter) ')']);
        
        %%%% Random font selection
        
        random_font = floor(rand * length(fontlist)) + 1;
        
        %%%% Creating text on the figure
        
        [im lh] = generateSample(letter,h,fontlist,random_font);
        
        %%%% Random affine transformation
        
        rx = (rand - 0.5) * maximum_x_rotation * pi / 180;
        ry = (rand - 0.5) * maximum_y_rotation * pi / 180;
        rz = (rand - 0.5) * maximum_z_rotation * pi / 180;
        
        Rx = vrrotvec2mat([1 0 0 rx]);
        Ry = vrrotvec2mat([0 1 0 ry]);
        Rz = vrrotvec2mat([0 0 1 rz]);
        
        tform = maketform('projective',Rx * Ry * Rz);
        
        [he wi] = size(im);
        
        transformed_im = imtransform(im,tform,'fillvalues',1,...
            'udata',[-wi/2+1 wi/2],'vdata',[-he/2+1 he/2],...
            'xdata',[-wi/2+1 wi/2],'ydata',[-he/2+1 he/2]);
        
        %%%% Random brightness offset for illumination invariance
        
        random_brightness = (rand - 0.5) * 1;
        
        transformed_im = transformed_im + random_brightness;
        
        %%%% Reduce the image to desired size here to preserve quality
        
        transformed_im = imresize(transformed_im,[training_image_height training_image_width]);
        
        %%%% Blend background image
        
        in = round((number_of_background_images - 1) * rand) + 1;
        bg = background_images{in};
        
        ix = round((size(bg,2) - 25) * rand) + 1;
        iy = round((size(bg,1) - 25) * rand) + 1;
        
        bi = bg(iy + (1:24), ix + (1:24));
        bi = double(bi);
        bi = bi/max(bi(:));
        ia = 0.5 * rand;
        
        alpha_blended_image = (1 - ia) * transformed_im + ia * bi;
        
        %%%% Save created image
        
        if letter <= 90
            imwrite(alpha_blended_image ,[database_path_name char(letter) '/' num2str(t) '.png']);
        else
            imwrite(alpha_blended_image ,[database_path_name char(letter) char(letter) '/' num2str(t) '.png']);
        end
        
    end
    
end