#include "ch.h"
#include "hal.h"
#include "chprintf.h"
#include "zigbee.h"
#include <string.h>

#define TX_API_ID 0x01

static SerialConfig serial_zigbee_cfg = {
  9600,
  0,
  USART_CR2_STOP1_BITS | USART_CR2_LINEN,
  0,
};

static BinarySemaphore bsem_zigbee;

static int init = 0;
static int ok = 0;

void loop_ok(char *str)
{
  chprintf((BaseChannel *) &SD1, "%s\n", str);
  
  while(!ok){
    chBSemReset(&bsem_zigbee,1);
    while(*str != 0) {
      chprintf((BaseChannel *) &SD3, "%c", *str);
      str++;
    }
    chBSemWait(&bsem_zigbee);
  }
  
  ok = 0;
}

static WORKING_AREA(zigbeeInitThread,1024);
static msg_t zigbee_init_thread(void *arg)
{
  (void)arg;
	
  chprintf((BaseChannel *)&SD1,"Starting zigbee config...\r\n");
	
  chThdSleep(1100);
  loop_ok("+++");
  chThdSleep(1000);
		
  //Determine the PAN ID
  loop_ok("ATID 1234\r");

  //Scan channel
  loop_ok("ATSC 10\r");

  //Allow everybody to join
  loop_ok("ATNJ FF\r");

  //ATBD7 for having a 115200 baud rate on the serial port	
  loop_ok("ATBD 7\r");

  //Enable frame based API	
  //loop_ok("ATAP 1\r");

  //Return the association status : 0 for a successful startup
  //loop_ok("ATAI\r");

  //Returns the coordinator address : 0
  //loop_ok("ATMY\r"); 

  //Leaving the configuration
  loop_ok("ATCN\r"); 

  chprintf((BaseChannel *) &SD1, "Ending zigbee config\r\n");
	
  sdStop(&SD3);
  sdStart(&SD3, NULL);
  		
  init = 1;
  
  return 0;
}

static WORKING_AREA(zigbeeMsgThread,1024);

int str_in_str(char *src, char *pattern, int length)
{
  int found = 0;
  int pattern_length = strlen(pattern);
  int i = 0;
  int j = 0;

  for(i = 0 ; i <= (length - pattern_length) ; ++i) {
    found = 1;
    for(j = 0 ; j < pattern_length ; ++j) {
      if(src[i+j] != pattern[j]) {
	found = 0;
	break;
      }
    }
    if(found)
      return 1;
  }

  return 0;
}

static msg_t zigbee_msg_thread(void *arg)
{
  (void)arg;
  char c;

  char buffer[80];
  int i = 0;
  int j = 0;
  int start_of_frame = 0;
  int reception_frame = 0;
  zigbee_frame_rx frame;
	
  while(1) {
    c=chIOGet((BaseChannel *) &SD3);
    
    if (!init) {
      buffer[i]=c;

      if (c == '\r') {
	buffer[i+1]=0;
			
	if (i >= 2)
	  if(str_in_str(buffer, "OK\r", i+2)) {
	    chBSemSignal(&bsem_zigbee);
	    chprintf((BaseChannel *) &SD1, "OK received !\r\n");
	    ok = 1;
	  }
						
	if (i >= 5)
	  if(str_in_str(buffer, "ERROR\r", i+2)) {
	    chBSemSignal(&bsem_zigbee);
	    chprintf((BaseChannel *) &SD1, "ERROR received !\r\n");
	    ok = 0;
	  }
	i = 0;
      }
      else if (i > 79) {
	chprintf((BaseChannel *) &SD1, "Zigbee buffer overflow !\r\n");
	i = 0;
      }
      else 
	i++;
    }
    else {
      /*
       * Frame reception
       */
      if (c == 0x7E) {
	start_of_frame = 1;
	reception_frame = 0;
      }
      else if (c == 0 && start_of_frame) {
	reception_frame = 1;
	i = 0;
	frame.size_data = 0;
	start_of_frame = 0;
      }
      else if (reception_frame) {
	if (i == 0)
	  frame.size_data = c;
	
	buffer[i] = c;
	i++;

	if (i == frame.size_data + 2) {
	  if (buffer[1] == 0x81) {
	    frame.address = buffer[2]*16*16 + buffer[3];					
	    frame.rssi = buffer[4];
	    frame.option = buffer[5];

	    for (j = 0; j < frame.size_data-5; j++)
	      frame.data[j] = buffer[j+6];

	    frame.checksum = buffer[frame.size_data + 1];
						
	    process_frame(frame);
	  }
	  else
	    reception_frame = 0;
	}
      }
    }
  }
  
  return 0;
}

void zigbee_init(void)
{
  /*
   * Semaphore initialization.
   */
  chBSemInit(&bsem_zigbee, 1);

  /*
   * Initializes the Serial driver 3.
   */
  sdStart(&SD3, &serial_zigbee_cfg);

  /*
   * Threads initialization.
   */
  chThdCreateStatic(zigbeeInitThread, sizeof(zigbeeInitThread), NORMALPRIO+1, zigbee_init_thread, NULL);
  chThdCreateStatic(zigbeeMsgThread, sizeof(zigbeeMsgThread), NORMALPRIO+1, zigbee_msg_thread, NULL);
}

void process_frame(zigbee_frame_rx frame)
{
  chprintf((BaseChannel *) &SD1, "Frame received %s\r\n", frame.data);
}

void zigbee_send_msg(char* str, uint16_t addr)
{
  zigbee_frame_tx frame;
  
  chprintf((BaseChannel *) &SD1, "Frame to be sent to %x\r\n", addr);
	
  frame = wrap_frame(/*frame number*/0, /*address*/addr, /*option*/0, str);
  chprintf((BaseChannel *) &SD3, "%c%c%c%c%c%c%c%c%s%c",
	   frame.header[0], 
	   frame.header[1], 
	   frame.header[2],
	   frame.id_package,
	   frame.id_frame,
	   (frame.address >> 8),
	   (frame.address & 0xFF),
	   frame.option,
	   str,
	   frame.checksum);
}

zigbee_frame_tx wrap_frame(uint8_t frame_number, uint16_t address, uint8_t option, char *data){
  zigbee_frame_tx frame;
  uint8_t size_data = 0;
  uint8_t checksum = 0;
  
  frame.id_package = 0x01; 
	
  checksum += TX_API_ID + frame_number + (address >> 8) + (address & 0xFF) + option;
	
  frame.header[0] = 0x7e;
  frame.header[1] = 0x0;
	
  while(*data != 0) {
    checksum += *data;
    size_data++;
    data++;
  }

  frame.header[2] = 5 + size_data;
	
  frame.id_frame = frame_number;
  frame.address = address;	 		
  frame.option = option;
  frame.checksum = 0xFF - checksum;	

  return frame;
}
