#include "ch.h"
#include "hal.h"
#include "rf_comm.h"
#include "lcd.h"
#include "chprintf.h"
#include "radio_reg_op.h"
#include "error.h"

#define SERIAL_DEBUG

static SerialConfig serial_rf_cfg = {
  57600,
  0,
  USART_CR2_STOP1_BITS | USART_CR2_LINEN,
  0,
};

uint8_t rf_comm_connect(uint8_t channel, uint8_t attempt_number, char error_msg[])
{
  uint8_t i;
  uint8_t result;
  uint8_t success = 0;
  
  lcd_clear_display(2);
  lcd_set_position(0, 0);
  lcd_puts("Channel ");
  lcd_write_uint(channel);
  lcd_set_position(1, 0);
  lcd_puts("Connection...");
	
  for (i=0; i<attempt_number; i++) {
    set_reg_b(REG_INTF_CH, channel);

    if (get_reg_b(REG_RWL_CH, &result) && result == channel)
      success++;
  }
  
  lcd_clear_display(1);
  lcd_set_position(1, 0);
  
  if (success)
    lcd_puts("Successful");
  else
    lcd_puts("Failure");

  send_error_msg("Robot not found...", error_msg);
  
  return success;
}

/*
 * Scanning function
 * local_channel : Number of the channel on which the communication with the robot has been established
 */
uint8_t rf_comm_scan(uint8_t from_channel)
{
  uint8_t remote_channel = 0xFF;
  uint8_t i;

#ifdef SERIAL_DEBUG
  chprintf((BaseChannel *) &SD1, "Scanning for robots...\r\n");
#endif
  
  for (i=from_channel; i<255; i++) { //TODO : Understand why it crashes with 256
#ifdef SERIAL_DEBUG
    chprintf((BaseChannel *) &SD1, "Channel %d\r\n", i);
#endif
    lcd_set_position(0, 12);
    lcd_write_uint((i - from_channel) * 100 / (256 - from_channel));
    lcd_putc('%');
    lcd_clear_display(1);
    lcd_set_position(1, 0);
    lcd_puts("Channel ");
    lcd_write_uint(i);
    
    set_reg_b(REG_INTF_CH, i);

    if (get_reg_b(REG_RWL_CH, &remote_channel) && i == remote_channel) {
#ifdef SERIAL_DEBUG
      chprintf((BaseChannel *) &SD1, "Robot on channel %d\r\n", i);
#endif
      return i;
    }
  }

  // If none of the channels have been responding
  return 0;
}



uint8_t rf_comm_synchronize(void)
{
  uint8_t c = 0xFF;
  uint8_t i;

  for (i=0; i<24; i++) {
    chprintf((BaseChannel *) &SD2, "%c", c);
  }

  chprintf((BaseChannel *) &SD2, "%c", 0xAA);
  c = chIOGet((BaseChannel *) &SD2);
#ifdef SERIAL_DEBUG
  chprintf((BaseChannel *) &SD1, "Valeur renvoyee par synchronize : 0x%x\r\n", c);
#endif
  return (c == 0xAA);
}


void rf_comm_init(void)
{
  /*
   * Resetting the PIC16
   */
  palClearPad(GPIOA, 4);
  chThdSleepMilliseconds(100);
  palSetPad(GPIOA, 4);
  chThdSleepMilliseconds(1000);
  palClearPad(GPIOA, 4);
  chThdSleepMilliseconds(100);
  palSetPad(GPIOA, 4);
  chThdSleepMilliseconds(1000);
  palClearPad(GPIOA, 4);
  chThdSleepMilliseconds(100);
  palSetPad(GPIOA, 4);
  chThdSleepMilliseconds(1500);
  
  /*
   * Starting communication
   */
  sdStart(&SD2, &serial_rf_cfg);
  
  uint8_t sync_number = 0;
  
  while (sync_number < 10) {
    if (!rf_comm_synchronize())
#ifdef SERIAL_DEBUG
      chprintf((BaseChannel *) &SD1, "Echec de la synchronisation.\r\n");
    else
#endif
      sync_number++;

    chThdSleepMilliseconds(100);
  }
  /*
  uint8_t result;
  chThdSleepMilliseconds(100);
  //if (get_reg_b(REG_INTF_VER, result))
  //chprintf((BaseChannel *) &SD1, "Version : %d\r\n", result);
  set_reg_b(REG_INTF_CH, 108);
  set_reg_b(REG_INTF_VER, 0xAA);
  chprintf((BaseChannel *) &SD1, "Here !\r\n", result);
  //*/
}
