#include "ch.h"
#include "hal.h"
#include "chprintf.h"
#include "remote_flash.h"
#include "radio_reg_op.h"
#include "ff.h"
#include "mmc.h"
#include "lcd.h"
#include "error.h"

#define SERIAL_DEBUG

static DWORD firmware_size;
static uint32_t firmware_position = 0;

void patch_data(uint8_t *data)
{
  uint32_t sum = 0;
  uint8_t i;

  // Clear the vector at 0x14 so it doesn't affect the checksum
  for (i = 0; i < 4; i++)
    data[i + 0x14] = 0;

  // Calculate a native checksum of the little endian vector table
  // TODO : check the ugly i++
  for (i = 0; i < (4 * 8);) {
    sum += data[i++];
    sum += data[i++] << 8;
    sum += data[i++] << 16;
    sum += data[i++] << 24;
  }

  // Negate the result and place in the vector at 0x14 as little endian
  sum = (uint32_t) (0 - sum);
  for (i = 0; i < 4; i++)
    data[i + 0x14] = (uint8_t) (sum >> (8 * i));
}

uint8_t prepare_sector(const uint32_t sector)
{
  uint8_t result;
  set_reg_dw(REGDW_BL_0, sector);  // sector to be prepared
  set_reg_b(REG_BL_CTRL, 2);       // prepare sector
  get_reg_b(REG_BL_CTRL, &result);
  return (result == 1);
}

uint8_t erase_sector(const uint32_t sector)
{
  uint8_t result;
  set_reg_dw(REGDW_BL_0, sector);  // sector to be erased
  set_reg_b(REG_BL_CTRL, 3);       // erase sector
  get_reg_b(REG_BL_CTRL, &result);
  return (result == 1);
}

uint8_t write_to_ram(uint8_t* data, const uint32_t addr, uint32_t size)
{
  uint8_t result;
  set_reg_dw(REGDW_BL_0, addr);  // start address
  set_reg_dw(REGDW_BL_1, size);  // data size
  set_reg_b(REG_BL_CTRL, 4);     // write RAM
  get_reg_b(REG_BL_CTRL, &result);

  if (result != 3) {
#ifdef SERIAL_DEBUG
    chprintf((BaseChannel *) &SD1, "Error while starting write to remote RAM.\r\n");
#endif
    return 0;
  }
  
  /*
   * Send the data block, and check status each 20 writes (= 20 encoded lines, which is when the checksum is sent to the ARM by the PIC)
   */
  uint32_t ctr = 0;
  uint32_t ws;
  
  while (size > 0) {
    ws = (size > 29 ? 29 : size);
    if (!set_reg_mb(REGMB_BL_DATA, data, ws)) 
      return 0;
    
    ctr++;
    data += ws;
    size -= ws;

    firmware_position += ws;
    lcd_set_position(0, 12);
    lcd_write_uint(firmware_position * 100 / firmware_size);
    lcd_putc('%');

    if (ctr == 20 && size != 0) {
      ctr = 0;
      get_reg_b(REG_BL_CTRL, &result);
      if (result != 3) {
#ifdef SERIAL_DEBUG
        chprintf((BaseChannel *) &SD1, "Error while writing data to remote RAM.\r\n");
#endif
        return 0;
      }
    }
  }

  // If writing was successful, REG_BL_CTRL will be 1 now
  get_reg_b(REG_BL_CTRL, &result);
  return (result == 1);
}

uint8_t ram_to_flash(const uint32_t src, const uint32_t dest, const uint32_t length)
{
  uint8_t result;
  set_reg_dw(REGDW_BL_0, src);      // source (RAM) address
  set_reg_dw(REGDW_BL_1, dest);     // destination (flash) address
  set_reg_dw(REGDW_BL_2, length);   // data length to copy
  set_reg_b(REG_BL_CTRL, 5);        // copy RAM to flash
  get_reg_b(REG_BL_CTRL, &result);
  return (result == 1);
}

/* 
 * For a given write size, returns the minimal valid flash write length
 * (see LPC manual, "Copy RAM to Flash" in Chapter 21; 8192 is valid only
 * on some devices)
 */
uint32_t flash_write_size(const uint32_t size)
{
  if (size <= 512)
    return 512;
  else if (size <= 1024)
    return 1024;
  else if (size <= 4096)
    return 4096;
  else
    return 8192;
}

uint8_t prog_buffer[8192];

uint8_t program_arm(char *filename, char error_msg[])
{
  uint32_t i;
  uint8_t result8;
  uint32_t result32;
  FIL firmware;
  
  lcd_clear_display(2);
  lcd_set_position(0, 0);
  lcd_puts("Flashing...");
  
  if (f_open(&firmware, filename, FA_OPEN_EXISTING | FA_READ) != FR_OK) {
#ifdef SERIAL_DEBUG
    chprintf((BaseChannel *) &SD1, "File not found !\r\n");
#endif
    send_error_msg("File not found !", error_msg);
    return 0;
  }

  firmware_size = f_size(&firmware);

  set_reg_b(REG_BL_CTRL, 1);
  i = get_reg_b(REG_BL_CTRL, &result8);

#ifdef SERIAL_DEBUG
  chprintf((BaseChannel *) &SD1, "Entering bootloader: result = %d.\r\n", i);
#endif
  if (i != 1) {
    get_reg_b(REG_BL_CTRL, &result8);
#ifdef SERIAL_DEBUG
    chprintf((BaseChannel *) &SD1, "Return code: %d\r\n", result8);
#endif
    send_error_msg("Failure in entering bootloader", error_msg);
    return 0;
  }

  get_reg_dw(REGDW_BL_0, &result32);
#ifdef SERIAL_DEBUG
  chprintf((BaseChannel *) &SD1, "Part ID: %08X\r\n", result32);
#endif

  int eof = 0;
  uint32_t sector = 0;
  uint32_t flash_addr = 0; // flash memory address to write to

  while (!eof) {
#ifdef SERIAL_DEBUG
    chprintf((BaseChannel *) &SD1, "Programming flash sector %d...\r\n", sector);
#endif

    // gets the size of the current flash sector to program
    uint32_t cnt = SectorTable_212x[sector];
    if (cnt == 0) {
#ifdef SERIAL_DEBUG
      chprintf((BaseChannel *) &SD1, "Program is too long to fit into the available memory of the ARM.\r\n");
#endif
      send_error_msg("Program is too long to fit into the available ARM memory", error_msg);
      return 0;
    }

    if (!prepare_sector(sector)) {
#ifdef SERIAL_DEBUG
      chprintf((BaseChannel *) &SD1, "Unable to prepare sector %d of flash memory for erasing.\r\n", sector);
#endif
      send_error_msg("Unable to prepare sector of flash memory for erasing", error_msg);
      return 0;
    }
    if (!erase_sector(sector)) {
#ifdef SERIAL_DEBUG
      chprintf((BaseChannel *) &SD1, "Unable to erase sector %d of flash memory.\r\n", sector);
#endif
      send_error_msg("Unable to erase sector of flash memory", error_msg);
      return 0;
    }
    
    UINT result = 0;

    // loads data from the file to fill the sector
    while (cnt > 0 && !eof) {
      f_read(&firmware, prog_buffer, sizeof(prog_buffer), &result);
      if (result == 0) {
	eof = 1;
	break;
      }

#ifdef SERIAL_DEBUG
      chprintf((BaseChannel *) &SD1, "#");
#endif

      if (flash_addr == 0)
        patch_data(prog_buffer);

      if (!write_to_ram(prog_buffer, LPC_RAMBASE, result)) {
#ifdef SERIAL_DEBUG
        chprintf((BaseChannel *) &SD1, "Unable to write data to remote RAM.\r\n");
#endif
	send_error_msg("Unable to write data to remote RAM", error_msg);
        return 0;
      }

      if (!prepare_sector(sector)) {
#ifdef SERIAL_DEBUG
        chprintf((BaseChannel *) &SD1,"Unable to prepare sector %d of flash memory for writing.\r\n", sector);
#endif
        send_error_msg("Unable to prepare sector of flash memory for writing", error_msg);
	return 0;
      }

      if (!ram_to_flash(LPC_RAMBASE, flash_addr, flash_write_size(result))) {
#ifdef SERIAL_DEBUG
        chprintf((BaseChannel *) &SD1,"Unable to write data to remote flash.\r\n");
#endif
	send_error_msg("Unable to write data to remote flash", error_msg);
        return 0;
      }
      
      flash_addr += flash_write_size(result);
      cnt -= result;
    }
    
#ifdef SERIAL_DEBUG
    chprintf((BaseChannel *) &SD1, "\r\n");
#endif
    sector++;
  }

  set_reg_b(REG_BL_CTRL, 0);
  get_reg_b(REG_BL_CTRL, &result8);
#ifdef SERIAL_DEBUG
  chprintf((BaseChannel *) &SD1, "Exiting bootloader: result = %d\r\n", result8);
#endif
  
  if (!get_reg_b(0, &result8)) {
#ifdef SERIAL_DEBUG
    chprintf((BaseChannel *) &SD1, "Communication error !\r\n");
#endif
    send_error_msg("Communication error !", error_msg);
    return 0;
  }
  else {
#ifdef SERIAL_DEBUG
    chprintf((BaseChannel *) &SD1, "Remote flashing successful !\r\n");
#endif
    lcd_set_position(1, 0);
    lcd_puts("Successful !");
    chThdSleepMilliseconds(1000);
  }

  return 1;
}

void remote_flash(void)
{
  uint8_t channel = 108;
  uint8_t result = 0;
 
  while (result != channel) {
    set_reg_b(REG_INTF_CH, channel);
         
    if (!get_reg_b(REG_RWL_CH, &result)) {
#ifdef SERIAL_DEBUG
      chprintf((BaseChannel *) &SD1, "Robot not found.\r\n");
#endif
      //return;
    }
#ifdef SERIAL_DEBUG
    else
      chprintf((BaseChannel *) &SD1, "Robot connected on channel %d\r\n", result);
#endif
  }
  
  if(!get_reg_b(REG_RWL_VER, &result)) {
#ifdef SERIAL_DEBUG
    chprintf((BaseChannel *) &SD1, "Unable to find the remote version\r\n");
#endif
    return;
  }
#ifdef SERIAL_DEBUG
  else if (result != REMOTE_VERSION) {
    chprintf((BaseChannel *) &SD1, "Version : 0x%x\r\n", result);
  }
#endif
  
  char error_msg[32];
  if (!mmc_init(error_msg))
    return;
  
  program_arm("main.bin", error_msg);
}
