#include "ch.h"
#include "hal.h"
#include "menu.h"
#include "lcd.h"
#include "chprintf.h"
#include "remote_flash.h"
#include "rf_comm.h"
#include "mmc.h"
#include "ff.h"
#include "error.h"
#include "joystick.h"
#include "salamandra.h"

//#define SERIAL_DEBUG

static BinarySemaphore bsem_menu;
static BinarySemaphore bsem_button;
static PRESSED_BUTTON pressed_button = NONE;
extern uint8_t input_button;
static MENU current_menu = MAIN;
static MENU previous_menu = MAIN;
MENU_ENTRY menu_main[CHOICE_NUMBER];
MENU_ENTRY menu_music[CHOICE_NUMBER];
MENU_ENTRY menu_battery[CHOICE_NUMBER];
MENU_ENTRY menu_connection[CHOICE_NUMBER];
MENU_ENTRY menu_radio[CHOICE_NUMBER];
MENU_ENTRY menu_scan[CHOICE_NUMBER];
MENU_ENTRY menu_select_robot[CHOICE_NUMBER];
MENU_ENTRY menu_in_progress[CHOICE_NUMBER];
MENU_ENTRY menu_robot[CHOICE_NUMBER];
MENU_ENTRY menu_robot_started[CHOICE_NUMBER];
MENU_ENTRY menu_error_state[CHOICE_NUMBER];
uint8_t robot_channel_list[CHOICE_NUMBER];

static BinarySemaphore bsem_lcd_line0;
static BinarySemaphore bsem_lcd_line1;
char lcd_sliding_text[2][SLIDING_TEXT_LENGTH];
static uint8_t lcd_line0_length = 0;
static uint8_t lcd_line1_length = 0;
static uint8_t lcd_line0_display = 0;
static uint8_t lcd_line1_display = 0;
static uint8_t lcd_position0 = 0;
static uint8_t lcd_position1 = 0;

uint8_t offset = 1;

static WORKING_AREA(lcdLine0Thread, 256);
static msg_t lcd_line0_thread(void *arg)
{
  (void)arg;
  uint16_t waiting_time;

  while(1) {
    chBSemWait(&bsem_lcd_line0);
    
    if (lcd_line0_display) {
      lcd_clear_display(0);
      lcd_set_position(0, 0);
      if (lcd_line0_length <= 16)
	lcd_puts(lcd_sliding_text[0]);
      else {
	lcd_puts(&lcd_sliding_text[0][lcd_position0]);
	lcd_position0 = (lcd_position0 + 1) % (lcd_line0_length - 15);
      }
    }
    
    chBSemSignal(&bsem_lcd_line0);
    waiting_time = (lcd_position0 < 1) ? 500 : 1000;
    chThdSleepMilliseconds(waiting_time);
  }
}

static WORKING_AREA(lcdLine1Thread, 512);
static msg_t lcd_line1_thread(void *arg)
{
  (void)arg;
  uint16_t waiting_time;
  
  while(1) {
    chBSemWait(&bsem_lcd_line1);

    if (lcd_line1_display) {
      lcd_clear_display(1);
      lcd_set_position(1, 0);
      if (lcd_line1_length <= 16)
	lcd_puts(lcd_sliding_text[1]);
      else {
	lcd_puts(&lcd_sliding_text[1][lcd_position1]);
	lcd_position1 = (lcd_position1 + 1) % (lcd_line1_length - 15);
      }
    }
    
    chBSemSignal(&bsem_lcd_line1);
    waiting_time = (lcd_position1 < 1) ? 500 : 1000;
    chThdSleepMilliseconds(waiting_time);
  }
}

static WORKING_AREA(inputButtonThread, 256);
static msg_t input_button_thread(void *arg)
{
  (void)arg;
  
  while(1) {
    chBSemWait(&bsem_button);
    
    if (input_button == 3)
      pressed_button = UP;
    else if (input_button == 4)
      pressed_button = DOWN;
    else if (input_button == 1)
      pressed_button = LEFT;
    else if (input_button == 2)
      pressed_button = RIGHT;
    else
      pressed_button = NONE;

    if (input_button == 5) {
      offset = (offset == 1) ? 3 : 1;
      input_button = 0;
    }
    
    chThdSleepMilliseconds(100);

    if (pressed_button != NONE)
      chBSemSignal(&bsem_menu);
    
    if (pressed_button == NONE)
      chBSemSignal(&bsem_button);
  }
  
  return 0;
}

static WORKING_AREA(menuThread, 2048+512);
static msg_t menu_thread(void *arg)
{
  (void)arg;
  uint8_t current_menu_display = 0;
  uint8_t selected_channel = 0;
  uint8_t mmc_connected = 0;
  char error_msg[ERROR_MSG_MAX_LENGTH];
  char current_path[128];
  uint8_t path_length = 2;
  DIR dir;
  FILINFO fno;
  int res;
  uint8_t file_number = 0;
  uint8_t i;
  uint8_t fname_length = 0;

  while(1) {
    chBSemSignal(&bsem_button);
    chBSemWait(&bsem_menu);
    input_button = 0;
    
#ifdef SERIAL_DEBUG
    if (pressed_button == UP)
      chprintf((BaseChannel *)&SD1,"Up\r\n");
    else if (pressed_button == DOWN)
      chprintf((BaseChannel *)&SD1,"Down\r\n");
    else if (pressed_button == LEFT)
      chprintf((BaseChannel *)&SD1,"Left\r\n");
    else if (pressed_button == RIGHT)
      chprintf((BaseChannel *)&SD1,"Right\r\n");
    else
      chprintf((BaseChannel *)&SD1,"WTF\r\n");

    //chprintf((BaseChannel *)&SD1,"Current displayed menu : %d\r\n", current_menu_display);
#endif

    if (current_menu == MAIN) {
      lcd_clear_display(2);
      lcd_set_position(0, 0);
      lcd_puts("   MAIN MENU    ");
      
      chBSemWait(&bsem_lcd_line1);
      lcd_line1_display = 1;
      lcd_position1 = 0;
      lcd_copy_str_in_buffer(menu_main[current_menu_display].menu_buffer, lcd_sliding_text[1], &lcd_line1_length);
      chBSemSignal(&bsem_lcd_line1);
      
      menu_do_pressed_button(menu_main, &current_menu_display);
      selected_channel = 0;
    }
    else if (current_menu == MUSIC) {
      lcd_clear_display(2);
      lcd_set_position(0, 0);
      lcd_puts("   MUSIC MENU   ");
      
      chBSemWait(&bsem_lcd_line1);
      lcd_line1_display = 1;
      lcd_position1 = 0;
      lcd_copy_str_in_buffer(menu_music[current_menu_display].menu_buffer, lcd_sliding_text[1], &lcd_line1_length);
      chBSemSignal(&bsem_lcd_line1);
      
      menu_do_pressed_button(menu_music, &current_menu_display);
    }
    else if (current_menu == BATTERY) {
      lcd_clear_display(2);
      lcd_set_position(0, 0);
      lcd_puts("  BATTERY MENU  ");
      
      chBSemWait(&bsem_lcd_line1);
      lcd_line1_display = 1;
      lcd_position1 = 0;
      lcd_copy_str_in_buffer(menu_battery[current_menu_display].menu_buffer, lcd_sliding_text[1], &lcd_line1_length);
      chBSemSignal(&bsem_lcd_line1);
      
      menu_do_pressed_button(menu_battery, &current_menu_display);
    }
    else if (current_menu == CONNECTION) {
      lcd_clear_display(2);
      lcd_set_position(0, 0);
      lcd_puts("CONNECTION MENU ");
      
      chBSemWait(&bsem_lcd_line1);
      lcd_line1_display = 1;
      lcd_position1 = 0;
      lcd_copy_str_in_buffer(menu_connection[current_menu_display].menu_buffer, lcd_sliding_text[1], &lcd_line1_length);
      chBSemSignal(&bsem_lcd_line1);
      selected_channel = 0;
      
      menu_do_pressed_button(menu_connection, &current_menu_display);
    }
    else if (current_menu == RADIO) {
      lcd_clear_display(2);
      lcd_set_position(0, 0);
      lcd_puts("   RADIO MENU   ");
      
      chBSemWait(&bsem_lcd_line1);
      lcd_line1_display = 1;
      lcd_position1 = 0;
      lcd_copy_str_in_buffer(menu_radio[current_menu_display].menu_buffer, lcd_sliding_text[1], &lcd_line1_length);
      chBSemSignal(&bsem_lcd_line1);
      
      menu_do_pressed_button(menu_radio, &current_menu_display);
    }
    else if (current_menu == SELECT_CHANNEL) {
      lcd_clear_display(2);
      lcd_set_position(0, 0);
      lcd_puts(" SELECT CHANNEL ");
      lcd_set_position(1, 0);
      lcd_write_uint(selected_channel);

      if (pressed_button == UP) {
	selected_channel = (selected_channel - 1) % 255;
	chBSemSignal(&bsem_menu);
      }
      else if (pressed_button == DOWN) {
	selected_channel = (selected_channel + 1) % 255;
	chBSemSignal(&bsem_menu);
      }
      else if (pressed_button == LEFT) {
	current_menu = RADIO;
	previous_menu = CONNECTION;
	current_menu_display = 0;
	selected_channel = 0;
	chBSemSignal(&bsem_menu);
      }
      else if (pressed_button == RIGHT) {
	lcd_clear_display(2);
	lcd_set_position(0, 0);
	lcd_puts("Channel ");
	lcd_write_uint(selected_channel);
	lcd_set_position(1, 0);
	lcd_puts("Connection...");

	rf_comm_init();
	
	if (rf_comm_connect(selected_channel, CONNECTION_ATTEMPT, error_msg))
	  current_menu = ROBOT;
	else
	  current_menu = ERROR_STATE;

	previous_menu = SELECT_CHANNEL;
	current_menu_display = 0;
	chBSemSignal(&bsem_menu);
      }

      pressed_button = NONE;
      
    }
    else if (current_menu == SCAN) {
      if (selected_channel == 0 || (current_menu_display == 1 && pressed_button == RIGHT)) {
	lcd_clear_display(2);
	lcd_set_position(0, 0);
	lcd_puts("Scanning...");
	
	rf_comm_init();
	
	selected_channel = rf_comm_scan(selected_channel);

	if (selected_channel == 0) {
	  send_error_msg("No robot found...", error_msg);
	  current_menu = ERROR_STATE;
	  previous_menu = RADIO;
	}
	
	pressed_button = NONE;
	current_menu_display = 0;
	chBSemSignal(&bsem_menu);
      }
      else if (current_menu_display == 2 && pressed_button == RIGHT) {   //Scan again
	selected_channel = 0;
	pressed_button = NONE;
	current_menu_display = 0;
	chBSemSignal(&bsem_menu);
      }
      else {
	lcd_clear_display(2);
	lcd_set_position(0, 0);
	lcd_puts("Robot on ch ");
	lcd_write_uint(selected_channel);


	chBSemWait(&bsem_lcd_line1);
	lcd_line1_display = 1;
	lcd_position1 = 0;
	lcd_copy_str_in_buffer(menu_scan[current_menu_display].menu_buffer, lcd_sliding_text[1], &lcd_line1_length);
	chBSemSignal(&bsem_lcd_line1);

	menu_do_pressed_button(menu_scan, &current_menu_display);
      }
    }
    else if (current_menu == SELECT_ROBOT) {
      lcd_clear_display(2);
      lcd_set_position(0, 0);
      lcd_puts("  SELECT ROBOT  ");

      chBSemWait(&bsem_lcd_line1);
      lcd_line1_display = 1;
      lcd_position1 = 0;
      lcd_copy_str_in_buffer(menu_select_robot[current_menu_display].menu_buffer, lcd_sliding_text[1], &lcd_line1_length);
      chBSemSignal(&bsem_lcd_line1);

      if (pressed_button == RIGHT)
	selected_channel = robot_channel_list[current_menu_display];

      menu_do_pressed_button(menu_select_robot, &current_menu_display);
    }
    else if (current_menu == IN_PROGRESS) {
      lcd_clear_display(2);
      lcd_set_position(0, 0);
      lcd_puts(" IN PROGRESS... ");

      chBSemWait(&bsem_lcd_line1);
      lcd_line1_display = 1;
      lcd_position1 = 0;
      lcd_copy_str_in_buffer(menu_in_progress[current_menu_display].menu_buffer, lcd_sliding_text[1], &lcd_line1_length);
      chBSemSignal(&bsem_lcd_line1);
      
      menu_do_pressed_button(menu_in_progress, &current_menu_display);
    }
    else if (current_menu == ROBOT) {
      lcd_clear_display(2);
      lcd_set_position(0, 0);
      lcd_puts("   ROBOT MENU   ");
      
      chBSemWait(&bsem_lcd_line1);
      lcd_line1_display = 1;
      lcd_position1 = 0;
      lcd_copy_str_in_buffer(menu_robot[current_menu_display].menu_buffer, lcd_sliding_text[1], &lcd_line1_length);
      chBSemSignal(&bsem_lcd_line1);
      
      menu_do_pressed_button(menu_robot, &current_menu_display);
    }
    else if (current_menu == ROBOT_STARTING) {
      if (salamandra_init(error_msg)) {
	current_menu = ROBOT_STARTED;
	previous_menu = ROBOT_STARTED;
	pressed_button = NONE;
	current_menu_display = 0;
	chBSemSignal(&bsem_menu);
      }
      else {
	current_menu = ERROR_STATE;
	previous_menu = ROBOT;
	pressed_button = NONE;
	current_menu_display = 0;
	chBSemSignal(&bsem_menu);
      }
    }
    else if (current_menu == ROBOT_STARTED) {
      lcd_clear_display(2);
      lcd_set_position(0, 0);
      lcd_puts(" ROBOT STARTED  ");
      
      chBSemWait(&bsem_lcd_line1);
      lcd_line1_display = 1;
      lcd_position1 = 0;
      lcd_copy_str_in_buffer(menu_robot_started[current_menu_display].menu_buffer, lcd_sliding_text[1], &lcd_line1_length);
      chBSemSignal(&bsem_lcd_line1);
      
      if (pressed_button == RIGHT)
	salamandra_stop();
      
      menu_do_pressed_button(menu_robot_started, &current_menu_display);
    }
    else if (current_menu == FLASHING) {
      previous_menu = ROBOT;
      
      if (!mmc_connected) {
	mmc_connected = mmc_init(error_msg);
	
	if (!mmc_connected) {
	  current_menu = ERROR_STATE;
	  previous_menu = FLASHING;
	  current_menu_display = 0;
	  chBSemSignal(&bsem_menu);
	}
	else {
	  current_path[0] = '0';
	  current_path[1] = ':';
	  current_path[2] = 0;
	  file_number = 0;
	  path_length = 2;
	}
      }
      
      if (mmc_connected) {
	lcd_clear_display(2);
	lcd_set_position(0, 0);
	lcd_puts("CHOOSE FIRMWARE");
	
	f_opendir(&dir, current_path);

	for (i=0; i<=file_number; i++)
	  res = f_readdir(&dir, &fno);

	if(res != FR_OK || fno.fname[0]==0) {
	  if (file_number == 0) {                                  //Case of an empty folder
	    if (pressed_button == LEFT) {
	      if (path_length == 2) {                                 //If already at root, go back
		previous_menu = CONNECTION;
		current_menu = ROBOT;
		current_menu_display = 0;
		file_number = 0;
		chBSemSignal(&bsem_menu);
	      }
	      else {                                                  //Go to previous folder
		while (current_path[path_length] != '/')
		  path_length--;
		
		current_path[path_length] = 0;
		file_number = 0;
		chBSemSignal(&bsem_menu);
	      }
	    }
	    else {
	      lcd_set_position(1, 0);
	      lcd_puts("Empty folder");
	    }
	  }
	  else {                                                   //End of the folder
	    file_number--;
	    chBSemSignal(&bsem_menu);
	  }
	}
	else {
	  lcd_set_position(1, 0);
	  lcd_puts(fno.fname);
	  if (fno.fattrib & AM_DIR)
	    lcd_putc('/');

	  fname_length = 0;
	  while (fno.fname[fname_length] != 0)
	    fname_length++;
	  
	  if (pressed_button == UP && file_number > 0) {
	    file_number--;
	    chBSemSignal(&bsem_menu);
	  }
	  else if (pressed_button == DOWN) {
	    file_number++;
	    chBSemSignal(&bsem_menu);
	  }
	  else if (pressed_button == LEFT) {
	    if (path_length == 2) {                                 //If already at root, go back
	      previous_menu = CONNECTION;
	      current_menu = ROBOT;
	      current_menu_display = 0;
	      file_number = 0;
	      chBSemSignal(&bsem_menu);
	    }
	    else {                                                  //Go to previous folder
	      while (current_path[path_length] != '/')
		path_length--;
	      
	      current_path[path_length] = 0;
	      file_number = 0;
	      chBSemSignal(&bsem_menu);
	    }
	  }
	  else if (pressed_button == RIGHT) {
	    if (fno.fattrib & AM_DIR) {                             //Get in a folder
	      current_path[path_length] = '/';
	      for (i=0; i<fname_length; i++)
		current_path[path_length+i+1] = fno.fname[i];
	      
	      path_length += fname_length + 1;
	      current_path[path_length] = 0;
	      file_number = 0;
	      chBSemSignal(&bsem_menu);
	    }
	    else if (fno.fname[fname_length-4] == '.' &&
		     fno.fname[fname_length-3] == 'B' &&
		     fno.fname[fname_length-2] == 'I' &&
		     fno.fname[fname_length-1] == 'N') {            //Flash if the extension is .bin
	      
	      
	      current_path[path_length] = '/';
	      for (i=0; i<fname_length; i++)
		current_path[path_length+i+1] = fno.fname[i];
	      
	      path_length += fname_length + 1;
	      current_path[path_length] = 0;

	      if (program_arm(current_path, error_msg)) {
		previous_menu = CONNECTION;
		current_menu = ROBOT;
	      }
	      else {
		previous_menu = FLASHING;
		current_menu = ERROR_STATE;
	      }
	      
	      current_path[0] = '0';
	      current_path[1] = ':';
	      current_path[2] = 0;
	      file_number = 0;
	      path_length = 2;
	      current_menu_display = 0;
	      chBSemSignal(&bsem_menu);
	    }
	  }

	  pressed_button = NONE;
	}
      }
    } 
    else if (current_menu == ERROR_STATE) {
      uint8_t i;
      
      if (!(error_msg[0] == 'E' &&
	    error_msg[1] == 'R' &&
	    error_msg[2] == 'R')) {
	for (i=ERROR_MSG_MAX_LENGTH-8; i>0; i--)
	  error_msg[i+7] = error_msg[i];
     
	error_msg[7] = error_msg[0];
	error_msg[0] = 'E';
	error_msg[1] = 'R';
	error_msg[2] = 'R';
	error_msg[3] = 'O';
	error_msg[4] = 'R';
	error_msg[5] = ':';
	error_msg[6] = ' ';
	error_msg[ERROR_MSG_MAX_LENGTH-1] = 0;
      }
      
      chBSemWait(&bsem_lcd_line0);
      lcd_line0_display = 1;
      lcd_position0 = 0;
      lcd_copy_str_in_buffer(error_msg, lcd_sliding_text[0], &lcd_line0_length);
      chBSemSignal(&bsem_lcd_line0);
      
      chBSemWait(&bsem_lcd_line1);
      lcd_line1_display = 1;
      lcd_position1 = 0;
      lcd_copy_str_in_buffer(menu_error_state[current_menu_display].menu_buffer, lcd_sliding_text[1], &lcd_line1_length);
      chBSemSignal(&bsem_lcd_line1);
      
      menu_do_pressed_button(menu_error_state, &current_menu_display);
    }  
  }
  
  return 0;
}

/*
 * Definition of the entry of a menu
 */
void menu_entry_define(MENU_ENTRY *menu_entry, char *str, MENU next_menu)
{
  uint8_t i = 0;
  
  while (*str != 0) {
    (*menu_entry).menu_buffer[i] = *str;
    str++;
    i++;
  }
  
  (*menu_entry).menu_buffer[i] = *str;
  (*menu_entry).next_menu = next_menu;
}

/*
 * Action according to the pressed_button
 */
void menu_do_pressed_button(MENU_ENTRY menu_entry[], uint8_t *current_menu_entry)
{
  if (pressed_button == UP) {
    do {
      *current_menu_entry = (*current_menu_entry + CHOICE_NUMBER - 1) % CHOICE_NUMBER;
    } while (menu_entry[*current_menu_entry].menu_buffer[0] == 0);
    chBSemSignal(&bsem_menu);
  }
  else if (pressed_button == DOWN) {
    do {
      *current_menu_entry = (*current_menu_entry + CHOICE_NUMBER + 1) % CHOICE_NUMBER;
    } while (menu_entry[*current_menu_entry].menu_buffer[0] == 0);
    chBSemSignal(&bsem_menu);
  }
  else if (pressed_button == LEFT) {
    current_menu = previous_menu;
    previous_menu = MAIN;
    *current_menu_entry = 0;
    lcd_line0_display = 0;
    lcd_line1_display = 0;
    chBSemSignal(&bsem_menu);
  }
  else if (pressed_button == RIGHT) {
    previous_menu = current_menu;
    current_menu = menu_entry[*current_menu_entry].next_menu;
    *current_menu_entry = 0;
    lcd_line0_display = 0;
    lcd_line1_display = 0;
    chBSemSignal(&bsem_menu);
  }

  pressed_button = NONE;
}

void lcd_copy_str_in_buffer(char *str, char buffer[], uint8_t *str_length)
{
  *str_length = 0;
  
  while (*str != 0) {
    buffer[*str_length] = *str;
    (*str_length)++;
    str++;
  }

  buffer[*str_length] = 0;
}

void menu_init(void)
{
  uint8_t i;
  
  lcd_init();
  lcd_clear_display(2);
  lcd_set_position(0, 0);
  lcd_puts("    Hello !     ");
  lcd_set_position(1, 0);
  lcd_puts("   I'm UReCon   ");
  chThdSleepMilliseconds(1000);
  
  /*
   * Semaphore initialization.
   */
  chBSemInit(&bsem_menu, 0);
  chBSemInit(&bsem_button, 0);
  chBSemInit(&bsem_lcd_line0, 0);
  chBSemInit(&bsem_lcd_line1, 0);
  
  /*
   * Menu state initialization
   */
  current_menu = MAIN;
  previous_menu = MAIN;

  /*
   * Menu content initialization
   */
  //MAIN
  menu_entry_define(&menu_main[0], "Connect to a robot", CONNECTION);
  menu_entry_define(&menu_main[1], "Battery state", BATTERY);
  menu_entry_define(&menu_main[2], "Play music", MUSIC);
  
  for (i=3; i<CHOICE_NUMBER; i++)
    menu_entry_define(&menu_main[i], "", MAIN);

  //MUSIC
  menu_entry_define(&menu_music[0], "Play mario", IN_PROGRESS);
  menu_entry_define(&menu_music[1], "Back to main menu", MAIN);

  for (i=2; i<CHOICE_NUMBER; i++)
    menu_entry_define(&menu_music[i], "", MAIN);

  //BATTERY
  menu_entry_define(&menu_battery[0], "Battery level", IN_PROGRESS);
  menu_entry_define(&menu_battery[1], "Current consumption", IN_PROGRESS);
  menu_entry_define(&menu_battery[2], "Back to main menu", MAIN);
  
  for (i=3; i<CHOICE_NUMBER; i++)
    menu_entry_define(&menu_battery[i], "", MAIN);

  //CONNECTION
  menu_entry_define(&menu_connection[0], "Radio connection", RADIO);
  menu_entry_define(&menu_connection[1], "Zigbee connection", IN_PROGRESS);
  menu_entry_define(&menu_connection[2], "Back to main menu", MAIN);
  
  for (i=3; i<CHOICE_NUMBER; i++)
    menu_entry_define(&menu_connection[i], "", MAIN);

  //RADIO
  menu_entry_define(&menu_radio[0], "Select a channel", SELECT_CHANNEL);
  menu_entry_define(&menu_radio[1], "Scan for robots", SCAN);
  menu_entry_define(&menu_radio[2], "Select a robot", SELECT_ROBOT);
  menu_entry_define(&menu_radio[3], "Back to main menu", MAIN);
  
  for (i=4; i<CHOICE_NUMBER; i++)
    menu_entry_define(&menu_radio[i], "", MAIN);

  //SCAN
  menu_entry_define(&menu_scan[0], "Connect to the robot", SELECT_CHANNEL);
  menu_entry_define(&menu_scan[1], "Continue the scan", SCAN);
  menu_entry_define(&menu_scan[2], "Scan again", SCAN);
  menu_entry_define(&menu_scan[3], "Back to main menu", MAIN);
  
  for (i=4; i<CHOICE_NUMBER; i++)
    menu_entry_define(&menu_scan[i], "", MAIN);

  //SELECT_ROBOT
  menu_entry_define(&menu_select_robot[0], "Salamandra", SELECT_CHANNEL);
  menu_entry_define(&menu_select_robot[1], "Back to main menu", MAIN);
  
  robot_channel_list[0] = 108;
  robot_channel_list[1] = 0;

  for (i=2; i<CHOICE_NUMBER; i++)
    menu_entry_define(&menu_select_robot[i], "", MAIN);

  //IN_PROGRESS
  menu_entry_define(&menu_in_progress[0], "Back to main menu", MAIN);
  
  for (i=1; i<CHOICE_NUMBER; i++)
    menu_entry_define(&menu_in_progress[i], "", MAIN);
  
  //ROBOT
  menu_entry_define(&menu_robot[0], "Start the robot", ROBOT_STARTING);
  menu_entry_define(&menu_robot[1], "Flash the robot", FLASHING);
  menu_entry_define(&menu_robot[2], "Back to main menu", MAIN);

  for (i=3; i<CHOICE_NUMBER; i++)
    menu_entry_define(&menu_robot[i], "", MAIN);

  //ROBOT STARTED
  menu_entry_define(&menu_robot_started[0], "Stop the robot", ROBOT);
  menu_entry_define(&menu_robot_started[1], "Back to main menu", MAIN);

  for (i=2; i<CHOICE_NUMBER; i++)
    menu_entry_define(&menu_robot_started[i], "", MAIN);
  
  //ERROR STATE
  menu_entry_define(&menu_error_state[0], "Back to main menu", MAIN);
  
  for (i=1; i<CHOICE_NUMBER; i++)
    menu_entry_define(&menu_error_state[i], "", MAIN);

  /*
   * Threads initialization.
   */
  chThdCreateStatic(inputButtonThread, sizeof(inputButtonThread), NORMALPRIO+1, input_button_thread, NULL);
  chThdCreateStatic(menuThread, sizeof(menuThread), NORMALPRIO+1, menu_thread, NULL);
  chThdCreateStatic(lcdLine0Thread, sizeof(lcdLine0Thread), NORMALPRIO+1, lcd_line0_thread, NULL);
  chThdCreateStatic(lcdLine1Thread, sizeof(lcdLine1Thread), NORMALPRIO+1, lcd_line1_thread, NULL);
}
