#include "ch.h"
#include "hal.h"
#include "lcd.h"

#define lcd_addr 0b0111011

/* 
 * I2C1 Configuration
 */
static const I2CConfig i2cfg1 = {
  OPMODE_I2C,
  400000,
  FAST_DUTY_CYCLE_2,
};

static msg_t status = RDY_OK;
static systime_t tmo = TIME_INFINITE;

/*
 * Buffers
 */
static uint8_t lcd_rx_data[LCD_RX_DEPTH];
static uint8_t lcd_tx_data[LCD_TX_DEPTH];

/*
 * LCD Set cursor position
 */
void lcd_set_position(uint8_t line, uint8_t position)
{
  i2cStart(&I2CD1, &i2cfg1);
  
  lcd_tx_data[0] = 0x00;
  lcd_tx_data[1] = 0x80 | (0x40 * (line % 2) + position);

  i2cAcquireBus(&I2CD1);
  status = i2cMasterTransmitTimeout(&I2CD1, lcd_addr, lcd_tx_data, 2, lcd_rx_data, 0, tmo);
  i2cReleaseBus(&I2CD1);

  i2cStop(&I2CD1);
}

/*
 * LCD clear display : 0 for the first line, 1 for the second line, anything else for both lines
 */
void lcd_clear_display(uint8_t line)
{
  int i;
  
  if (line != 1) {
    //Set the cursor on the first spot of the first line
    lcd_set_position(0, 0);

    i2cStart(&I2CD1, &i2cfg1);

    //Write blank characters all along the line
    lcd_tx_data[0] = 0x40;
    for(i = 1; i < 17; i++)
      lcd_tx_data[i] = 0x91;

    i2cAcquireBus(&I2CD1);
    status = i2cMasterTransmitTimeout(&I2CD1, lcd_addr, lcd_tx_data, 17, lcd_rx_data, 0, tmo);
    i2cReleaseBus(&I2CD1);

    i2cStop(&I2CD1);
  }

  if (line != 0) {
    //Set the cursor on the first spot of the second line
    lcd_set_position(1, 0);

    i2cStart(&I2CD1, &i2cfg1);

    //Write blank characters all along the line
    lcd_tx_data[0] = 0x40;
    for(i = 1; i < 17; i++)
      lcd_tx_data[i] = 0x91;

    i2cAcquireBus(&I2CD1);
    status = i2cMasterTransmitTimeout(&I2CD1, lcd_addr, lcd_tx_data, 17, lcd_rx_data, 0, tmo);
    i2cReleaseBus(&I2CD1);

    i2cStop(&I2CD1);
  }
}

/*
 * Writes a character on the LCD screen
 */
void lcd_putc(char c)
{
  i2cStart(&I2CD1, &i2cfg1);
  
  lcd_tx_data[0] = 0x40;
  lcd_tx_data[1] = c + 0x80;

  i2cAcquireBus(&I2CD1);
  status = i2cMasterTransmitTimeout(&I2CD1, lcd_addr, lcd_tx_data, 2, lcd_rx_data, 0, tmo);
  i2cReleaseBus(&I2CD1);
    
  i2cStop(&I2CD1);
}

/*
 * Writes a string on the LCD screen
 */
void lcd_puts(char *s)
{
  while (*s != 0) {
    lcd_putc(*s);
    s++;
  }
}

/*
 * LCD screen initialization
 */
void lcd_init(void)
{
  i2cInit();
  i2cStart(&I2CD1, &i2cfg1);
  
  //Pause added just to be safe
  chThdSleepMilliseconds(100);
  
  //Configure LCD : Comments for G. Cuendet
  lcd_tx_data[0] = 0x00;  // Control byte -> CO: 0, RS: 0
  lcd_tx_data[1] = 0x34;  // FUNCTION SET -> DL: 8 bits, M:  two line, SL: 1:18, H: normal instruction set
  lcd_tx_data[2] = 0x0F;  // DISPLAY_CTL  -> D: on, C: on, B:on
  lcd_tx_data[3] = 0x06;  // ENTRY_MODE_SE-> I_D: increments to the right, S: no shift
  
  lcd_tx_data[4] = 0x35;  // FUNCTION SET -> DL: 8 bits, M: 16 by two line, SL: 1:18, H: extended instruction set
  lcd_tx_data[5] = 0x07;  // DISP_CONF    -> P: right to left, Q: bottom to top
  lcd_tx_data[6] = 0x43;  // HV_GEN       -> HV do not use (11)
  lcd_tx_data[7] = 0xE8;  // VLCD_SET     -> set Vlcd to 5V, store to VB
  lcd_tx_data[8] = 0xA8;  // VLCD_SET     -> set Vlcd to 5V(A8), store to VA
  lcd_tx_data[9] = 0x08;
  
  lcd_tx_data[10] = 0x34; // FUNCTION SET -> DL: 8 bits, M:  two line, SL: 1:18, H: normal instruction set
  lcd_tx_data[11] = 0x80; // SET_CGRAM    -> DDRAM Address set to 0x00
  lcd_tx_data[12] = 0x01; // CLEAR_DISPLAY
  
  //Sending the initialization instructions for the LCD screen
  i2cAcquireBus(&I2CD1);
  status = i2cMasterTransmitTimeout(&I2CD1, lcd_addr, lcd_tx_data, 13, lcd_rx_data, 0, tmo);
  i2cReleaseBus(&I2CD1);

  chThdSleepMilliseconds(20);
  i2cStop(&I2CD1);
  
  lcd_clear_display(2);
  lcd_set_position(0, 0);
  lcd_puts("Hello World !");
}

/*
 * Writes a number on the lcd screen at the current cursor position
 */
void lcd_write_uint(uint32_t number)
{
  uint32_t temp = number;
  uint32_t new_number = 1;
  
  do {
    new_number = new_number * 10 + (temp % 10);
    temp /= 10;
  } while (temp != 0);

  while (new_number != 1) {
    lcd_putc('0'+ (new_number % 10));
    new_number /= 10;
  }
}
