#include "ch.h"
#include "hal.h"
#include "joystick.h"
#include "chprintf.h"
#include "radio_reg_op.h"
#include "salamandra.h"

#define ADC_JOY1_NUM_CHANNELS   2
#define ADC_JOY1_BUF_DEPTH      8

#define ADC_JOY2_NUM_CHANNELS   2
#define ADC_JOY2_BUF_DEPTH      8

//#define SERIAL_DEBUG

static adcsample_t samples1[ADC_JOY1_NUM_CHANNELS * ADC_JOY1_BUF_DEPTH];
static adcsample_t samples2[ADC_JOY2_NUM_CHANNELS * ADC_JOY2_BUF_DEPTH];

static uint8_t thd_exit = 0;
static msg_t exit_msg;
extern uint8_t offset;

/*
 * ADC conversion group for Joystick 1.
 * Mode:        Linear, 8 samples of 2 channels, SW triggered.
 * Channels:    IN0, IN1.
 */
static const ADCConversionGroup adcjoycfg1 = {
  FALSE,
  ADC_JOY1_NUM_CHANNELS,
  NULL,
  NULL,
  0, 0,                         /* CR1, CR2 */
  ADC_SMPR1_SMP_AN11(ADC_SAMPLE_41P5) | ADC_SMPR1_SMP_AN10(ADC_SAMPLE_41P5),
  0,                            /* SMPR2 */
  ADC_SQR1_NUM_CH(ADC_JOY1_NUM_CHANNELS),
  0,                            /* SQR2 */
  ADC_SQR3_SQ2_N(ADC_CHANNEL_IN1) | ADC_SQR3_SQ1_N(ADC_CHANNEL_IN0)
};

/*
 * ADC conversion group for Joystick 2.
 * Mode:        Linear, 8 samples of 2 channels, SW triggered.
 * Channels:    IN5, IN6.
 */
static const ADCConversionGroup adcjoycfg2 = {
  FALSE,
  ADC_JOY1_NUM_CHANNELS,
  NULL,
  NULL,
  0, 0,                         /* CR1, CR2 */
  ADC_SMPR1_SMP_AN11(ADC_SAMPLE_41P5) | ADC_SMPR1_SMP_AN10(ADC_SAMPLE_41P5),
  0,                            /* SMPR2 */
  ADC_SQR1_NUM_CH(ADC_JOY2_NUM_CHANNELS),
  0,                            /* SQR2 */
  ADC_SQR3_SQ2_N(ADC_CHANNEL_IN6) | ADC_SQR3_SQ1_N(ADC_CHANNEL_IN5)
};

static WORKING_AREA(salamandraThread, 512);
static msg_t salamandra_thread(void *arg)
{
  (void)arg;

  uint8_t joystick1_x;
  uint16_t joystick1_y;
  float body_params[8] = {1, 5, 0.5, 1.3, 0, 33.333, 60, 0};
  float legs_params[8] = {1, 3, 0.2, 0.6, 0, 33.333, 60, 0};
  uint8_t freq_body;
  uint8_t ampl_body;
  uint8_t freq_legs;
  uint8_t ampl_legs;
  float correct_freq = 2.326;
  
  while(1) {
    if (thd_exit) {
      chThdExit(exit_msg);
      adcStop(&ADCD1);
    }    

    chThdSleepMilliseconds(20);
    
    //adcAcquireBus(&ADCD1);
    adcStartConversion(&ADCD1, &adcjoycfg1, samples1, ADC_JOY1_BUF_DEPTH);
    //adcReleaseBus(&ADCD1);
    
    joystick1_x = 15 - (samples1[0] * 15 / 4098);
    joystick1_y = samples1[1];
    
    freq_body = (uint8_t) (correct_freq * 63.8 * (body_params[2] + (body_params[3] - body_params[2]) / (body_params[1] - body_params[0]) * ((joystick1_y*2/4098)+offset - body_params[0])));
    ampl_body = (uint8_t) (4.2058 * (body_params[5] + (body_params[6] - body_params[5]) / (body_params[1] - body_params[0]) * ((joystick1_y*2/4098)+offset - body_params[0])));
    freq_legs = (uint8_t) (correct_freq * 63.8 * (legs_params[2] + (legs_params[3] - legs_params[2]) / (legs_params[1] - legs_params[0]) * ((joystick1_y*2/4098)+offset - legs_params[0])));
    ampl_legs = (uint8_t) (4.2058 * (legs_params[5] + (legs_params[6] - legs_params[5]) / (legs_params[1] - legs_params[0]) * ((joystick1_y*2/4098)+offset - legs_params[0])));

    if (offset == 3) {
      freq_legs = (uint8_t) (correct_freq * 63.8 * legs_params[4]);
      ampl_legs = (uint8_t) (4.2058 * legs_params[7]);
    }

    set_reg_b(REGB_PHASE, 255/2);
    set_reg_b(REGB_TURN, ((joystick1_x << 4) | 8));
    set_reg_b(REGB_AMPL_L, ampl_legs);
    set_reg_b(REGB_AMPL_B, ampl_body);
    set_reg_b(REGB_FREQ_L, freq_legs);
    set_reg_b(REGB_FREQ_B, freq_body);

#ifdef SERIAL_DEBUG
    chprintf((BaseChannel *) &SD1, "Joystick 1 : %x \r\n", joystick1_x);
#endif
  }

  return 0;
}

void joystick_init(void)
{
  /*
   * Activates the ADC1 driver.
   */
  adcStart(&ADCD1, NULL);
  chThdSleepMilliseconds(100);

  /*
   * Threads initialization.
   */
  thd_exit = 0;
  chThdCreateStatic(salamandraThread, sizeof(salamandraThread), NORMALPRIO+1, salamandra_thread, NULL);
}

void joystick_stop(void)
{
  thd_exit = 1;
}
