#ifndef _REMOTE_FLASH_H_
#define _REMOTE_FLASH_H_

#define LPC_RAMSTART     0x40000000   // the physical start address of the SRAM
#define LPC_RAMBASE      0x40000200   // the base address where downloading starts
#define LPC_FLASHMASK    0xFFC00000   // mask for file loading; 22 bits = 4 MB

#define REG_MODE         0            // mode register on remote firmware
#define REG_INTF_VER     0x3C0        // Radio interface's version
#define REG_INTF_CH      0x3C1        // Local (radio interface) channel number
#define REG_RWL_VER      0x3E0        // Remote radio firmware's version
#define REG_RWL_CH       0x3E1        // Remote channel number (= REG_INTF_CH, if the communication is set)
#define REG_REM_VERSION  0x3E0        // remote radio firmware version
#define REG_BL_CTRL      0x3E2        // bootloader control register
#define REGDW_PART_ID    0x3E0        // (read) part ID (when in BL mode)
#define REGDW_BL_0       0x3E0        // (write) bootloader parameter 0
#define REGDW_BL_1       0x3E1        // (write) bootloader parameter 1
#define REGDW_BL_2       0x3E2        // (write) bootloader parameter 2
#define REGMB_BL_DATA    0x3E2        // (write) bootloader program data

#define INTF_VERSION     5            // expected firmware versions
#define REMOTE_VERSION   0x49         // 48 = PIC bootloader support, 49 = LPC ARM

typedef struct {
  uint32_t id;
  uint32_t Product;
  uint32_t FlashSize;          /* in kiB, for informational purposes only */
  uint32_t RAMSize;            /* in kiB, for informational purposes only */
  uint32_t FlashSectors;       /* total number of sectors */
  uint32_t MaxCopySize;        /* maximum size that can be copied to Flash in a single command */
  const uint32_t *SectorTable; /* pointer to a sector table with constant the sector sizes */
} LPC_DEVICE_TYPE;

static const uint32_t SectorTable_210x[] =
{
  8192, 8192, 8192, 8192, 8192, 8192, 8192, 8192,
  8192, 8192, 8192, 8192, 8192, 8192, 8192
};

static const uint32_t SectorTable_2103[] =
{
  4096, 4096, 4096, 4096, 4096, 4096, 4096, 4096
};

static const uint32_t SectorTable_2109[] =
{
  8192, 8192, 8192, 8192, 8192, 8192, 8192, 8192
};

static const uint32_t SectorTable_211x[] =
{
  8192, 8192, 8192, 8192, 8192, 8192, 8192, 8192,
  8192, 8192, 8192, 8192, 8192, 8192, 8192,
};

static const uint32_t SectorTable_212x[] =
{
  8192, 8192, 8192, 8192, 8192, 8192, 8192, 8192,
  65536, 65536, 8192, 8192, 8192, 8192, 8192, 8192, 8192
};

// Used for devices with 500K (LPC2138 and LPC2148) and
// for devices with 504K (1 extra 4k block at the end)
static const uint32_t SectorTable_213x[] =
{
   4096,  4096,  4096,  4096,  4096,  4096,  4096,  4096,
  32768, 32768, 32768, 32768, 32768, 32768, 32768, 32768,
  32768, 32768, 32768, 32768, 32768, 32768,  4096,  4096,
   4096,  4096,  4096,  4096
};

static LPC_DEVICE_TYPE LPCtypes[] =
{
  { 0, 0, 0 },  /* unknown */
  { 0x0004FF11, 2103,  32,  8,  8, 4096, SectorTable_2103 },
  { 0xFFF0FF12, 2104, 128, 16, 15, 8192, SectorTable_210x },
  { 0xFFF0FF22, 2105, 128, 32, 15, 8192, SectorTable_210x },
  { 0xFFF0FF32, 2106, 128, 64, 15, 8192, SectorTable_210x },
  { 0x0201FF01, 2109,  64,  8,  8, 4096, SectorTable_2109 },
  { 0x0101FF12, 2114, 128, 16, 15, 8192, SectorTable_211x },
  { 0x0201FF12, 2119, 128, 16, 15, 8192, SectorTable_211x },
  { 0x0101FF13, 2124, 256, 16, 17, 8192, SectorTable_212x },
  { 0x0201FF13, 2129, 256, 16, 17, 8192, SectorTable_212x },
  { 0x0002FF01, 2131,  32,  8,  8, 4096, SectorTable_213x },
  { 0x0002FF11, 2132,  64, 16,  9, 4096, SectorTable_213x },
  { 0x0002FF12, 2134, 128, 16, 11, 4096, SectorTable_213x },
  { 0x0002FF23, 2136, 256, 32, 15, 4096, SectorTable_213x },
  { 0x0002FF25, 2138, 512, 32, 27, 4096, SectorTable_213x },
  { 0x0402FF01, 2141,  32,  8,  8, 4096, SectorTable_213x },
  { 0x0402FF11, 2142,  64, 16,  9, 4096, SectorTable_213x },
  { 0x0402FF12, 2144, 128, 16, 11, 4096, SectorTable_213x },
  { 0x0402FF23, 2146, 256, 40, 15, 4096, SectorTable_213x },
  { 0x0402FF25, 2148, 512, 40, 27, 4096, SectorTable_213x },
  { 0x0301FF13, 2194, 256, 16, 17, 8192, SectorTable_212x },
  { 0x0301FF12, 2210,   0, 16,  0, 8192, SectorTable_211x }, /* table is a "don't care" */
  { 0x0401FF12, 2212, 128, 16, 15, 8192, SectorTable_211x },
  { 0x0601FF13, 2214, 256, 16, 17, 8192, SectorTable_212x },
  /*            2290; same id as the LPC2210 */
  { 0x0401FF13, 2292, 256, 16, 17, 8192, SectorTable_212x },
  { 0x0501FF13, 2294, 256, 16, 17, 8192, SectorTable_212x },
  { 0x00000000, 2361, 128, 34, 11, 4096, SectorTable_213x },
  { 0x00000000, 2362, 128, 34, 11, 4096, SectorTable_213x },
  { 0x1600F902, 2364, 128, 34, 11, 4096, SectorTable_213x },
  { 0x1600E823, 2365, 256, 58, 15, 4096, SectorTable_213x },
  { 0x1600F923, 2366, 256, 58, 15, 4096, SectorTable_213x },
  { 0x1600E825, 2367, 512, 58, 15, 4096, SectorTable_213x },
  { 0x1600F925, 2368, 512, 58, 28, 4096, SectorTable_213x },
  { 0x1700E825, 2377, 512, 58, 28, 4096, SectorTable_213x },
  { 0x1700FD25, 2378, 512, 58, 28, 4096, SectorTable_213x },
  { 0x1800F935, 2387, 512, 98, 28, 4096, SectorTable_213x },
  { 0x1800FF35, 2388, 512, 98, 28, 4096, SectorTable_213x },
  { 0x1500FF35, 2458, 512, 98, 28, 4096, SectorTable_213x },
  { 0x1600FF30, 2460,   0, 98,  0, 4096, SectorTable_213x },
  { 0x1600FF35, 2468, 512, 98, 28, 4096, SectorTable_213x },
  { 0x1701FF30, 2470,   0, 98,  0, 4096, SectorTable_213x },
  { 0x1701FF35, 2478, 512, 98, 28, 4096, SectorTable_213x }
};

void patch_data(uint8_t *data);
uint8_t prepare_sector(const uint32_t sector);
uint8_t erase_sector(const uint32_t sector);
uint8_t write_to_ram(uint8_t* data, const uint32_t addr, uint32_t size);
uint8_t ram_to_flash(const uint32_t src, const uint32_t dest, const uint32_t length);
uint32_t flash_write_size(const uint32_t size);
uint8_t program_arm(char *filename, char error_msg[]);
void remote_flash(void);

#endif
